<?php
require_once __DIR__ . '/config/db.php';

// Get token from URL
$token = isset($_GET['token']) ? sanitize($_GET['token']) : '';

if (empty($token)) {
    die('Invalid access. QR code token is required.');
}

// Get employee by token
try {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT id, name, department FROM employees WHERE qr_token = ?");
    $stmt->execute([$token]);
    $employee = $stmt->fetch();
    
    if (!$employee) {
        die('Invalid token. Employee not found.');
    }
    
    // Check today's attendance status
    $employeeId = $employee['id'];
    $today = date('Y-m-d');
    $stmt = $pdo->prepare("SELECT check_in_time, check_out_time FROM attendance WHERE employee_id = ? AND date = ?");
    $stmt->execute([$employeeId, $today]);
    $todayAttendance = $stmt->fetch();
    
    $hasCheckedIn = $todayAttendance && $todayAttendance['check_in_time'] ? true : false;
    $hasCheckedOut = $todayAttendance && $todayAttendance['check_out_time'] ? true : false;
    $checkInTime = $hasCheckedIn ? $todayAttendance['check_in_time'] : null;
    $checkOutTime = $hasCheckedOut ? $todayAttendance['check_out_time'] : null;
    
} catch (PDOException $e) {
    error_log("Attendance page error: " . $e->getMessage());
    die('Error loading employee data.');
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <!-- CSS FILE LINKS -->
    <link rel="stylesheet" href="/css/global.css">
    <link rel="stylesheet" href="/css/attendence.css">
    <link rel="stylesheet" href="/css/alert.css">
    <!-- WEBSITE TITLE -->
    <title>Employee Attendance</title>
</head>

<body>

    <div class="attendenceScreen">
        <h1 id="employee-name"><?php echo htmlspecialchars($employee['name']); ?></h1>
        <p id="employee-department"><?php echo htmlspecialchars($employee['department']); ?></p>
        
        <?php if ($hasCheckedIn): ?>
            <div style="margin: 15px 0; padding: 10px; background: #d4edda; border: 1px solid #c3e6cb; border-radius: 5px; color: #155724;">
                <strong>✓ Checked In:</strong> <?php echo htmlspecialchars($checkInTime); ?>
            </div>
        <?php endif; ?>
        
        <?php if ($hasCheckedOut): ?>
            <div style="margin: 15px 0; padding: 10px; background: #d1ecf1; border: 1px solid #bee5eb; border-radius: 5px; color: #0c5460;">
                <strong>✓ Checked Out:</strong> <?php echo htmlspecialchars($checkOutTime); ?>
            </div>
        <?php endif; ?>
        
        <form action="" class="attendenceBox" id="attendanceForm">
            
            <div class="radio-container">
                <div class="radio-wrapper">
                    <label class="radio-button" style="<?php echo $hasCheckedIn ? 'opacity: 0.5; pointer-events: none;' : ''; ?>">
                        <input id="check-in" name="checkingTime" type="radio" value="check-in" required <?php echo $hasCheckedIn ? 'disabled' : ''; ?>>
                        <span class="radio-checkmark"></span>
                        <span class="radio-label">Check In <?php echo $hasCheckedIn ? '(Already Checked In)' : ''; ?></span>
                    </label>
                </div>

                <div class="radio-wrapper">
                    <label class="radio-button" style="<?php echo ($hasCheckedOut || !$hasCheckedIn) ? 'opacity: 0.5; pointer-events: none;' : ''; ?>">
                        <input id="check-out" name="checkingTime" type="radio" value="check-out" required <?php echo ($hasCheckedOut || !$hasCheckedIn) ? 'disabled' : ''; ?>>
                        <span class="radio-checkmark"></span>
                        <span class="radio-label">Check Out <?php echo $hasCheckedOut ? '(Already Checked Out)' : (!$hasCheckedIn ? '(Check In First)' : ''); ?></span>
                    </label>
                </div>

                <button type="submit" <?php echo ($hasCheckedIn && $hasCheckedOut) ? 'disabled style="opacity: 0.5;"' : ''; ?>>Submit</button>

            </div>

        </form>
    </div>

    <div class="alert" id="alert">
        <div class="info">
            <div class="info__title" id="alert-message">lorem ipsum dolor sit amet</div>
            <button class="info__close" id="alert-close"><svg xmlns="http://www.w3.org/2000/svg" width="20" viewBox="0 0 20 20"
                    height="20">
                    <path fill="#393a37"
                        d="m15.8333 5.34166-1.175-1.175-4.6583 4.65834-4.65833-4.65834-1.175 1.175 4.65833 4.65834-4.65833 4.6583 1.175 1.175 4.65833-4.6583 4.6583 4.6583 1.175-1.175-4.6583-4.6583z">
                    </path>
                </svg></button>
        </div>
    </div>

    <script>
        const token = '<?php echo htmlspecialchars($token); ?>';
        let hasCheckedIn = <?php echo $hasCheckedIn ? 'true' : 'false'; ?>;
        let hasCheckedOut = <?php echo $hasCheckedOut ? 'true' : 'false'; ?>;
        
        // Attendance form handler
        document.getElementById('attendanceForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const action = document.querySelector('input[name="checkingTime"]:checked')?.value;
            
            if (!action) {
                showAlert('Please select Check-In or Check-Out', 'error');
                return;
            }
            
            // Disable submit button
            const submitBtn = this.querySelector('button[type="submit"]');
            submitBtn.disabled = true;
            submitBtn.textContent = 'Processing...';
            
            try {
                const response = await fetch('/api/attendance.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({ token, action })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showAlert(data.message + ' at ' + data.time, 'success');
                    
                    // Update UI based on response
                    if (data.hasCheckedIn) {
                        hasCheckedIn = true;
                        disableCheckIn();
                    }
                    if (data.hasCheckedOut) {
                        hasCheckedOut = true;
                        disableCheckOut();
                    }
                    
                    // Reset form
                    document.querySelectorAll('input[name="checkingTime"]').forEach(radio => {
                        radio.checked = false;
                    });
                    
                    // Reload page after 2 seconds to show updated status
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                } else {
                    showAlert(data.message, 'error');
                }
            } catch (error) {
                showAlert('Network error. Please try again.', 'error');
            } finally {
                submitBtn.disabled = false;
                submitBtn.textContent = 'Submit';
            }
        });
        
        function disableCheckIn() {
            const checkInRadio = document.getElementById('check-in');
            const checkInLabel = checkInRadio.closest('label');
            checkInRadio.disabled = true;
            checkInLabel.style.opacity = '0.5';
            checkInLabel.style.pointerEvents = 'none';
        }
        
        function disableCheckOut() {
            const checkOutRadio = document.getElementById('check-out');
            const checkOutLabel = checkOutRadio.closest('label');
            checkOutRadio.disabled = true;
            checkOutLabel.style.opacity = '0.5';
            checkOutLabel.style.pointerEvents = 'none';
        }
        
        // Alert functions
        function showAlert(message, type = 'info') {
            const alert = document.getElementById('alert');
            const alertMessage = document.getElementById('alert-message');
            alertMessage.textContent = message;
            alert.classList.add('show');
            
            setTimeout(() => {
                hideAlert();
            }, 5000);
        }
        
        function hideAlert() {
            const alert = document.getElementById('alert');
            alert.classList.remove('show');
        }
        
        document.getElementById('alert-close').addEventListener('click', hideAlert);
        document.getElementById('alert').classList.remove('show');
    </script>

</body>

</html>
